<?php
namespace Wizhou\Magasin;
use Wizhou\Magasin\OrderCustomer;
use Wizhou\Magasin\OrderShipping;
use Kirby\Toolkit\A;
use Kirby\Toolkit\Str;
use Kirby\Cms\Page;

class Order
{
    private $id;
    private $session;
    private $customer;
    private $lineItems;
    private $shipping;
    private $cart;
    private $orderNumber;
    private $total;
    private $subtotal;
    private $tax;

    public function __construct (
      $csrf, // Kirby csrf
      $sessionId, // Stripe Session ID
      $paymentMethod = 'stripe' // Payment method
    ) {
      if ($paymentMethod === 'stripe') {
        $this->retriveOrderDataFromStripe($csrf, $sessionId);
      }

      // Create order
      // Check if order is new
    }

    private function retriveOrderDataFromStripe($csrf, $sessionId) {
      payments()::setStripeApiKey();

      /* Retrive info from Stripe */
      $this->id = $sessionId;
      $this->session = \Stripe\Checkout\Session::retrieve($this->id);

      // Retrive items
      $this->lineItems = \Stripe\Checkout\Session::allLineItems($this->id);

      $customer = new OrderCustomer($this->session->customer_details);

      $this->customer = $customer;

      $shipping = new OrderShipping(
        $this->session->shipping_cost,
        $this->session->shipping_details,
        $this->session->shipping_options,
      );

      $this->shipping = $shipping;

      // Get order number and session ID
      $orderId = $this->id;
      $orderDate = date('c', $this->session->created);
      $orderNumber = 'ON-' . $this->session->created;
      $this->orderNumber = $orderNumber;

      $this->subtotal = currency()->toPrice($this->session->amount_subtotal);
      $this->total = currency()->toPrice($this->session->amount_total);

    }

    /* Return a stripe customer adress in one line */

    static function prettifyAddress ($adress) {
      $toJoin = [
        $adress->line1,
        $adress->line2,
        $adress->postal_code,
        $adress->city,
        $adress->state,
        $adress->country
      ];

      foreach ($toJoin as $id => $el) {
        if (!$el) unset($toJoin[$id]);
      }

      return A::join($toJoin, ' ');
    }

    public function session () {
      return $this->session;
    }

    public function lineItems () {
      return $this->lineItems;
    }

    public function customer () {
      return $this->customer;
    }

    public function shipping () {
      return $this->shipping;
    }

    public function orderNumber () {
      return $this->orderNumber;
    }

    public function getShippingTotal () {
      return $this->shipping()->total();
    }

    public function cart () {
      return $this->cart;
    }

    /* Create an order page if the checkout-success page appear
    for the first time and store the data of the page. The stock is then updated inside the page.create.after hook.  */

    public function createOrderPage(
      $checkoutCart // Collection with Stripe Product
    ) {
        $kirby = kirby();
        $site = site();

        $session = $this->session;
        $customer = $this->customer;
        $shipping = $this->shipping;
        $lineItems = $this->lineItems;
        $orderNumber = $this->orderNumber;

        /* Check if order page already exist */
        $orderPage = $site->page('orders/' . $orderNumber);
        if ($orderPage && $orderPage->exists()) {
          // Order Page already exist
          return false;
        }

        $products = [];

        foreach ($checkoutCart as $id => $product) {
          $uuid = $product->metadata->uuid;
          $lineItem = $lineItems->data[$id];

          $products[] = [
            'id' => $product->id,
            'subtotal' => currency()->toPrice($lineItem->amount_subtotal),
            'tax' => currency()->toPrice($lineItem->amount_tax),
            'total' => currency()->toPrice($lineItem->amount_total),
            'description' => $lineItem->description,
            'quantity' => $lineItem->quantity,
            'page' => $uuid
          ];
        }

        /* Create Order Page */
        $orderPage = [
          'slug' => $orderNumber,
          'parent' => $site->page('orders'),
          'template' => 'order',
          'content' => [
            'orderNumber' => $orderNumber,
            'orderDate' => $orderDate,
            'sessionId' => $orderId,
            'payementItent' => $session->payment_intent,
            'shippingRate' => $shipping->option()->id,
            'payementStatus' => $session->payment_status,
            'orderState' => 'pending',
            'subtotal' => $this->subtotal,
            'shippingTotal' => currency()->toPrice($shipping->total()),
            'total' => $this->total,
            'customer' => [
              'name' => $customer->name(),
              'email' => $customer->email(),
              'phone' => $customer->phone(),
              'city' => $customer->address()->city,
              'country' => $customer->address()->country,
              'line1' => $customer->address()->line1,
              'line2' => $customer->address()->line2,
              'postalCode' => $customer->address()->postal_code,
              'state' => $customer->address()->state,
            ],
            'shipping' => [
              'type' => $shipping->option()->display_name,
              'amountTotal' => currency()->toPrice($shipping->total()),
              'city' => $shipping->address()->city,
              'country' => $shipping->address()->country,
              'line1' => $shipping->address()->line1,
              'line2' => $shipping->address()->line2,
              'postalCode' => $shipping->address()->postal_code,
              'state' => $shipping->address()->state,
              'name' => $shipping->name(),
            ],
            'products' => $products,
            'session' => json_encode($session),
            'line_items' => json_encode($lineItems),
            'shippingData' => json_encode($shipping),
            'date' => date('Ymdhhsu', $session->created)
          ]
        ];

        $kirby->impersonate('kirby');
        $order = Page::create($orderPage);
        $order = $order->changeStatus('listed');
    }

    public function sendOrderEmail ($template) {

      // $template 'order-success' = order page is created and stripe payments is successful.
      $kirby = kirby();
      $site = site();

      $session = $this->session;
      $customer = $this->customer;
      $shipping = $this->shipping;
      $lineItems = $this->lineItems;

      $orderPage = $site->page('orders/' . $this->orderNumber);

      $subject = '';
      $cancelMessage = '';
      $refundMessage = '';

      switch ($template) {
        case 'order-success':
          $subject = 'Votre commande ' . $this->orderNumber . ' chez ' . $site->title();
          break;

        case 'order-accepted':
          $subject = 'Votre commande ' . $this->orderNumber . ' chez ' . $site->title() . ' est confirmée';
          break;

        case 'order-sent':
          $subject = 'Votre commande ' . $this->orderNumber . ' chez ' . $site->title() . ' à été expédiée';
          break;

        case 'order-canceled':
          $subject = 'Votre commande ' . $this->orderNumber . ' chez ' . $site->title() . ' à été annulée';
          $cancelMessage = $orderPage->cancelMessage();
          break;

        case 'order-refunded':
          $subject = 'Votre commande ' . $this->orderNumber . ' chez ' . $site->title() . ' à été remboursée';
          $refundMessage = $orderPage->refundMessage();
          break;

        default:
            throw new \Exception("There is no template selected", 1);
          break;
      }

      $kirby->email([
        'from' => $site->mail()->value(),
        'to' => $customer->email(),
        'subject' =>  $subject,
        'template' => $template,
        'data' => [
          'customerName' => $customer->name(),
          'siteEmail' => $site->mail(),
          'orderDate' => date('d/m/Y', $session->created),
          'amountTotal' => Currency()->prettify($session->amount_total, 'amount'),
          'billingAdress' => $customer->prettyAddress(),
          'shippingAdress' => $shipping->prettyAddress(),
          'shippingOptions' => $shipping->option()->display_name,
          'shippingPrice' => r($shipping->option()->fixed_amount, currency()->prettify($shipping->option()->fixed_amount->amount, 'amount')),
          'orderNumber' => $this->orderNumber,
          'products' => $orderPage->products()->toStructure(),
          'site' => $site,
          'line_items' => $lineItems,
          'session' => $session,
          'cancelMessage' => $cancelMessage
        ]
      ]);
    }
}
